<?php
/**
 * Mosallas Publisher
 * Handles article creation and publishing
 */

if (!defined('ABSPATH')) {
    exit;
}

class Mosallas_Publisher {
    
    /**
     * Store current article data for category selection
     */
    private $current_article_title = '';
    private $current_article_content = '';
    
    /**
     * Create and publish a post
     */
    public function create_post($data) {
        try {
            error_log('Mosallas Publisher: Starting post creation');
            error_log('Mosallas Publisher: Data keys - ' . implode(', ', array_keys($data)));
            
            // Store article data for category selection
            $this->current_article_title = $data['title'] ?? '';
            $this->current_article_content = $data['content'] ?? '';
            
            // Validate input data
            if (!is_array($data)) {
                throw new Exception('داده‌های ورودی نامعتبر است');
            }
            
            if (empty($data['title'])) {
                throw new Exception('عنوان مقاله خالی است');
            }
            
            if (empty($data['content'])) {
                throw new Exception('محتوای مقاله خالی است');
            }
            
            $featured_image_candidates = $this->get_featured_image_candidates($data);
            
            // Get or create category
            try {
                $category_ids = $this->get_or_create_category($data['topic'] ?? '');
                error_log('Mosallas Publisher: Category IDs: ' . print_r($category_ids, true));
                Mosallas_Logger::info('📁 دسته‌بندی تنظیم شد', array(
                    'category_ids' => $category_ids,
                    'topic' => $data['topic'] ?? ''
                ));
            } catch (Exception $cat_error) {
                error_log('Mosallas Publisher: Category creation failed - ' . $cat_error->getMessage());
                Mosallas_Logger::warning('⚠️ خطا در ایجاد دسته‌بندی (استفاده از دسته پیش‌فرض)', array(
                    'error' => $cat_error->getMessage()
                ));
                $category_ids = array(get_option('default_category', 1));
            }
            
            // Format the content
            try {
                error_log('Mosallas Publisher: Formatting content');
                Mosallas_Logger::info('📝 در حال قالب‌بندی محتوا');
                $formatted_content = $this->format_content($data);
            } catch (Exception $format_error) {
                error_log('Mosallas Publisher: Content formatting failed - ' . $format_error->getMessage());
                Mosallas_Logger::error('❌ خطا در قالب‌بندی محتوا', array(
                    'error' => $format_error->getMessage()
                ));
                throw new Exception('خطا در قالب‌بندی محتوا: ' . $format_error->getMessage());
            }
            
            // Prepare post data
            $post_data = array(
                'post_title' => sanitize_text_field($data['title']),
                'post_content' => $formatted_content,
                'post_excerpt' => !empty($data['abstract']) ? sanitize_textarea_field($data['abstract']) : '',
                'post_status' => 'publish',
                'post_author' => $this->get_author_id(),
                'post_category' => $category_ids,
                'post_type' => 'post'
            );
            
            error_log('Mosallas Publisher: Inserting post');
            Mosallas_Logger::info('💾 در حال ذخیره مقاله در وردپرس', array(
                'title' => $data['title']
            ));
            
            // Insert the post
            $post_id = wp_insert_post($post_data, true);
            
            if (is_wp_error($post_id)) {
                $error_message = $post_id->get_error_message();
                error_log('Mosallas Publisher: wp_insert_post failed - ' . $error_message);
                Mosallas_Logger::error('❌ خطا در ذخیره مقاله در پایگاه داده وردپرس', array(
                    'error' => $error_message,
                    'post_data' => $post_data
                ));
                throw new Exception('خطا در ذخیره مقاله: ' . $error_message);
            }
            
            error_log('Mosallas Publisher: Post created with ID: ' . $post_id);
            Mosallas_Logger::success('✅ مقاله در وردپرس ذخیره شد', array(
                'post_id' => $post_id
            ));
            
            $featured_image_id = 0;
            
            // Priority 1: Use featuredImageUrl or featured_image_url from payload
            $payload_featured_image_url = '';
            if (!empty($data['featuredImageUrl'])) {
                $payload_featured_image_url = $data['featuredImageUrl'];
            } elseif (!empty($data['featured_image_url'])) {
                $payload_featured_image_url = $data['featured_image_url'];
            }
            
            if (!empty($payload_featured_image_url)) {
                error_log('Mosallas Publisher: Using payload featured image URL: ' . $payload_featured_image_url);
                Mosallas_Logger::info('🖼️ استفاده از تصویر شاخص ارسال‌شده در payload');
                $featured_image_id = $this->assign_featured_image($post_id, array($payload_featured_image_url), $data['title']);
            }
            
            // Priority 2: If no payload image, try other candidates from data
            if (!$featured_image_id) {
                $featured_image_candidates = $this->get_featured_image_candidates($data);
                if (!empty($featured_image_candidates)) {
                    error_log('Mosallas Publisher: Using other featured image candidates from data');
                    Mosallas_Logger::info('🖼️ استفاده از سایر کاندیداهای تصویر شاخص');
                    $featured_image_id = $this->assign_featured_image($post_id, $featured_image_candidates, $data['title']);
                }
            }

            // Priority 3: Fallback to extracting images from article content
            if (!$featured_image_id && !empty($data['content'])) {
                error_log('Mosallas Publisher: No payload or data images found, extracting from content');
                Mosallas_Logger::info('🖼️ استخراج تصویر از محتوای مقاله');
                $fallback_candidates = $this->extract_image_sources_from_content($data['content']);
                if (!empty($fallback_candidates)) {
                    $featured_image_id = $this->assign_featured_image($post_id, $fallback_candidates, $data['title']);
                }
            }
            
            // Update SEO meta tags
            try {
                error_log('Mosallas Publisher: Updating SEO meta tags');
                Mosallas_Logger::info('🎯 در حال تنظیم تگ‌های سئو');
                $this->update_seo_meta($post_id, $data);
                Mosallas_Logger::success('✅ تگ‌های سئو تنظیم شدند');
            } catch (Exception $seo_error) {
                error_log('Mosallas Publisher: SEO meta update failed - ' . $seo_error->getMessage());
                Mosallas_Logger::warning('⚠️ خطا در تنظیم تگ‌های سئو', array(
                    'error' => $seo_error->getMessage()
                ));
            }
            
            // Add internal links (after post is created)
            if (!empty($data['existingPosts'])) {
                error_log('Mosallas Publisher: Adding internal links');
                $this->add_internal_links($post_id, $data['existingPosts']);
            }
            
            // Add custom meta data
            error_log('Mosallas Publisher: Adding custom meta data');
            update_post_meta($post_id, '_mosallas_generated', true);
            update_post_meta($post_id, '_mosallas_generated_date', current_time('mysql'));
            if (!empty($data['articleId'])) {
                update_post_meta($post_id, '_mosallas_article_id', sanitize_text_field($data['articleId']));
            }
            
            error_log('Mosallas Publisher: Post creation completed successfully');
            
            return $post_id;
            
        } catch (Exception $e) {
            error_log('Mosallas Publisher: Exception caught - ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Resolve ordered candidate URLs for the featured image.
     */
    private function get_featured_image_candidates($data) {
        $candidates = array();

        if (!is_array($data)) {
            return $candidates;
        }

        $keys = array(
            'featuredImageUrl',
            'featured_image_url',
            'featuredImageURL',
            'featured_image',
            'featuredImage',
            'coverImageUrl',
            'cover_image_url'
        );

        foreach ($keys as $key) {
            if (!isset($data[$key])) {
                continue;
            }

            $value = $data[$key];

            if (is_string($value) && !empty(trim($value))) {
                $candidates[] = trim($value);
            } elseif (is_array($value) && isset($value['url']) && !empty($value['url'])) {
                $candidates[] = trim($value['url']);
            }
        }

        // Flatten nested image arrays that may be provided
        if (isset($data['featuredImage']) && is_array($data['featuredImage'])) {
            if (isset($data['featuredImage']['src'])) {
                $candidates[] = trim($data['featuredImage']['src']);
            }
            if (isset($data['featuredImage']['source'])) {
                $candidates[] = trim($data['featuredImage']['source']);
            }
        }

        $unique = array();
        $final = array();
        foreach ($candidates as $candidate) {
            if (!is_string($candidate) || $candidate === '') {
                continue;
            }

            $candidate = html_entity_decode(trim($candidate));
            if ($candidate === '') {
                continue;
            }

            if (!preg_match('/^https?:\/\//i', $candidate)) {
                continue; // Ignore relative URLs
            }

            $hash = md5($candidate);
            if (isset($unique[$hash])) {
                continue;
            }

            $unique[$hash] = true;
            $final[] = $candidate;
        }

        return $final;
    }

    /**
     * Try assigning a featured image using the provided candidate URLs.
     */
    private function assign_featured_image($post_id, array $candidates, $title) {
        foreach ($candidates as $candidate_url) {
            try {
                $attachment_id = $this->upload_image($candidate_url, $title, $post_id);
                if ($attachment_id > 0) {
                    // Set the uploaded image as the post's featured image
                    $result = set_post_thumbnail($post_id, $attachment_id);
                    if ($result) {
                        Mosallas_Logger::success('🖼️ تصویر شاخص به مقاله اضافه شد', array(
                            'attachment_id' => $attachment_id,
                            'post_id' => $post_id,
                            'source' => $candidate_url
                        ));
                        return $attachment_id;
                    } else {
                        Mosallas_Logger::warning('⚠️ تصویر آپلود شد اما تنظیم به عنوان تصویر شاخص ناموفق بود', array(
                            'attachment_id' => $attachment_id,
                            'post_id' => $post_id,
                            'source' => $candidate_url
                        ));
                    }
                }
            } catch (Exception $e) {
                Mosallas_Logger::warning('⚠️ تلاش برای تنظیم تصویر شاخص ناموفق بود', array(
                    'error' => $e->getMessage(),
                    'source' => $candidate_url
                ));
            }
        }

        return 0;
    }

    /**
     * Extract image sources from article HTML content.
     */
    private function extract_image_sources_from_content($content) {
        $sources = array();

        if (empty($content) || !is_string($content)) {
            return $sources;
        }

        $dom = new DOMDocument();
        libxml_use_internal_errors(true);
        $loaded = $dom->loadHTML(mb_convert_encoding($content, 'HTML-ENTITIES', 'UTF-8'));
        libxml_clear_errors();

        if (!$loaded) {
            return $sources;
        }

        $images = $dom->getElementsByTagName('img');
        foreach ($images as $image) {
            $src = $image->getAttribute('src');
            if (empty($src)) {
                continue;
            }

            $src = html_entity_decode(trim($src));
            if (!preg_match('/^https?:\/\//i', $src)) {
                continue;
            }

            $sources[] = $src;
        }

        return $sources;
    }
    
    /**
     * Format article content with all sections
     */
    private function format_content($data) {
        try {
            $content = '';
            
            // Validate data is not null
            if (!is_array($data)) {
                error_log('Mosallas Publisher: Invalid data format for format_content - not an array');
                throw new Exception('داده‌های مقاله نامعتبر است');
            }
            
            // Add abstract/introduction as clean HTML
            if (!empty($data['abstract'])) {
                $content .= "\n\n" . '<p><strong>خلاصه:</strong></p>' . "\n";
                $content .= '<p>' . wp_kses_post($data['abstract']) . '</p>' . "\n\n";
            }
            
            // Add table of contents as clean HTML
            if (!empty($data['tableOfContents'])) {
                $toc = $data['tableOfContents'];
                
                // If it's a JSON string, decode it
                if (is_string($toc)) {
                    $decoded = json_decode($toc, true);
                    if (json_last_error() === JSON_ERROR_NONE) {
                        $toc = $decoded;
                    } else {
                        error_log('Mosallas Publisher: JSON decode error for tableOfContents - ' . json_last_error_msg());
                        $toc = array(); // Reset to empty array on error
                    }
                }
                
                if (is_array($toc) && !empty($toc)) {
                    $content .= '<h2>فهرست مطالب</h2>' . "\n";
                    $content .= '<ul>' . "\n";
                    foreach ($toc as $item) {
                        $heading = '';
                        $id = '';

                        if (is_string($item)) {
                            $heading = $item;
                        } elseif (is_array($item)) {
                            // Support structures such as { title: "...", id: "..." }
                            $heading = $item['title'] ?? $item['text'] ?? $item['label'] ?? '';
                            $id = $item['id'] ?? $item['anchor'] ?? '';
                        } elseif (is_object($item)) {
                            $heading = $item->title ?? $item->text ?? $item->label ?? '';
                            $id = isset($item->id) ? $item->id : (isset($item->anchor) ? $item->anchor : '');
                        }

                        if (!empty($heading)) {
                            if (!empty($id)) {
                                $content .= '<li><a href="#' . esc_attr($id) . '">' . esc_html($heading) . '</a></li>' . "\n";
                            } else {
                                $content .= '<li>' . esc_html($heading) . '</li>' . "\n";
                            }
                        }
                    }
                    $content .= '</ul>' . "\n\n";
                }
            }
            
            // Add main content as raw HTML (this is the main article content)
            if (!empty($data['content'])) {
                // Use the content as-is, but allow WordPress to handle it properly
                // Remove excessive filtering to preserve HTML structure for editing
                $main_content = $data['content'];
                
                // Only do basic security filtering, preserve HTML structure
                $allowed_html = wp_kses_allowed_html('post');
                
                // Add additional allowed tags for better content preservation
                $allowed_html['figure'] = array(
                    'class' => array(),
                    'id' => array(),
                    'style' => array()
                );
                $allowed_html['figcaption'] = array(
                    'class' => array(),
                    'id' => array()
                );
                $allowed_html['section'] = array(
                    'class' => array(),
                    'id' => array()
                );
                $allowed_html['article'] = array(
                    'class' => array(),
                    'id' => array()
                );
                
                $content .= wp_kses($main_content, $allowed_html) . "\n\n";
            } else {
                error_log('Mosallas Publisher: Missing content field');
                throw new Exception('محتوای مقاله خالی است');
            }
            
            // Add FAQ section as clean HTML
            if (!empty($data['faqSection'])) {
                $faqContent = $data['faqSection'];

                // If FAQ is provided as JSON, transform to HTML
                if (is_string($faqContent)) {
                    $decodedFaq = json_decode($faqContent, true);
                    if (json_last_error() === JSON_ERROR_NONE && is_array($decodedFaq)) {
                        $faqHtml = '';
                        foreach ($decodedFaq as $faqItem) {
                            if (is_array($faqItem)) {
                                $question = $faqItem['question'] ?? '';
                                $answer = $faqItem['answer'] ?? '';
                            } elseif (is_object($faqItem)) {
                                $question = $faqItem->question ?? '';
                                $answer = $faqItem->answer ?? '';
                            } else {
                                $question = '';
                                $answer = '';
                            }

                            if (!empty($question) || !empty($answer)) {
                                if (!empty($question)) {
                                    $faqHtml .= '<h3>' . esc_html($question) . '</h3>' . "\n";
                                }
                                if (!empty($answer)) {
                                    $faqHtml .= '<p>' . wp_kses_post($answer) . '</p>' . "\n";
                                }
                            }
                        }

                        if (!empty($faqHtml)) {
                            $faqContent = $faqHtml;
                        }
                    }
                }

                $content .= '<h2>سوالات متداول (FAQ)</h2>' . "\n";
                $content .= wp_kses_post($faqContent) . "\n";
            }
            
            return $content;
            
        } catch (Exception $e) {
            error_log('Mosallas Publisher: Exception in format_content - ' . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Upload image from URL to WordPress media library
     */
    private function upload_image($image_url, $title, $post_id = 0) {
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/media.php');
        require_once(ABSPATH . 'wp-admin/includes/image.php');

        $image_url = $this->normalize_remote_image_url($image_url);
        if (empty($image_url)) {
            Mosallas_Logger::warning('⚠️ آدرس تصویر شاخص نامعتبر است');
            return 0;
        }

        // Download the image
        $temp_file = download_url($image_url, 30);
        
        if (is_wp_error($temp_file)) {
            error_log('Mosallas: Failed to download image - ' . $temp_file->get_error_message());
            Mosallas_Logger::warning('⚠️ دانلود تصویر شاخص ناموفق بود', array(
                'error' => $temp_file->get_error_message(),
                'source' => $image_url
            ));
            return 0;
        }
        
        // Try to resolve file extension from URL path (ignoring query params)
        $parsed_url = wp_parse_url($image_url);
        $url_path = isset($parsed_url['path']) ? $parsed_url['path'] : '';
        $file_extension = strtolower(pathinfo($url_path, PATHINFO_EXTENSION));

        // Initial sanitization to drop unexpected characters
        if (!empty($file_extension)) {
            $file_extension = preg_replace('/[^a-z0-9]/', '', $file_extension);
        }

        $mime_type = '';
        $allowed_extensions = array('jpg', 'jpeg', 'png', 'gif', 'webp', 'bmp', 'tiff', 'svg', 'avif', 'heic', 'heif');

        if (empty($file_extension) || !in_array($file_extension, $allowed_extensions, true)) {
            // Attempt to detect via HTTP headers
            $head_response = wp_safe_remote_head($image_url, array('timeout' => 15));
            if (is_wp_error($head_response) || 200 !== wp_remote_retrieve_response_code($head_response)) {
                // Some servers block HEAD; try a lightweight GET request
                $head_response = wp_safe_remote_get($image_url, array(
                    'timeout' => 15,
                    'redirection' => 5,
                    'headers' => array('Range' => 'bytes=0-0')
                ));
            }

            if (!is_wp_error($head_response)) {
                $content_type_header = wp_remote_retrieve_header($head_response, 'content-type');
                if (!empty($content_type_header)) {
                    $mime_type = is_array($content_type_header) ? end($content_type_header) : $content_type_header;
                    $detected_ext = $this->guess_extension_from_mime($mime_type);
                    if (!empty($detected_ext)) {
                        $file_extension = $detected_ext;
                    }
                }
            }
        }

        // As a final fallback, rely on WordPress file type detection
        if (empty($file_extension) || !in_array($file_extension, $allowed_extensions, true)) {
            $filetype = wp_check_filetype_and_ext($temp_file, wp_basename($url_path));
            if (!empty($filetype['ext'])) {
                $file_extension = $filetype['ext'];
            }
            if (empty($mime_type) && !empty($filetype['type'])) {
                $mime_type = $filetype['type'];
            }
        }

        if (empty($file_extension)) {
            $file_extension = 'jpg';
        }

        if (empty($mime_type)) {
            $mime_type = $this->guess_mime_from_extension($file_extension);
        }

        // Build a safe filename from title (handles non-Latin titles)
        $base_name = sanitize_file_name($title);
        if (empty($base_name)) {
            $base_name = 'mosallas-featured-image';
        }
        $file_name = $base_name . '-' . time() . '.' . $file_extension;
        
        // Prepare file array
        $file = array(
            'name' => $file_name,
            'tmp_name' => $temp_file,
            'type' => $mime_type
        );
        
        // Upload the file
        $attachment_id = media_handle_sideload($file, $post_id, $title);
        
        // Clean up temp file
        if (is_wp_error($attachment_id)) {
            @unlink($temp_file);
            error_log('Mosallas: Failed to upload image - ' . $attachment_id->get_error_message());
            Mosallas_Logger::warning('⚠️ بارگذاری تصویر شاخص ناموفق بود', array(
                'error' => $attachment_id->get_error_message(),
                'source' => $image_url
            ));
            return 0;
        }
        
        // Set alt text
        update_post_meta($attachment_id, '_wp_attachment_image_alt', $title);
        
        // Generate attachment metadata (sizes) to ensure proper display
        $file_path = get_attached_file($attachment_id);
        if ($file_path) {
            $metadata = wp_generate_attachment_metadata($attachment_id, $file_path);
            if (!is_wp_error($metadata) && !empty($metadata)) {
                wp_update_attachment_metadata($attachment_id, $metadata);
            } else {
                error_log('Mosallas: Failed to generate attachment metadata for image ID ' . $attachment_id);
            }
        }
        
        return $attachment_id;
    }

    /**
     * Normalize remote image URLs to avoid download issues.
     */
    private function normalize_remote_image_url($url) {
        if (!is_string($url)) {
            return '';
        }

        $url = html_entity_decode(trim($url));
        if ($url === '') {
            return '';
        }

        // Replace spaces and control characters
        $url = preg_replace('/[\s]+/u', '%20', $url);

        $parsed = wp_parse_url($url);
        if (empty($parsed) || empty($parsed['host'])) {
            return $url;
        }

        $host = strtolower($parsed['host']);
        $local_hosts = array('localhost', '127.0.0.1', 'host.docker.internal');

        if (in_array($host, $local_hosts, true)) {
            $replacement_host = apply_filters('mosallas_minio_public_host', 's3api.mosallas.ir', $host, $parsed);
            if (!empty($replacement_host) && is_string($replacement_host)) {
                $parsed['scheme'] = 'https';
                $parsed['host'] = $replacement_host;
                unset($parsed['port']);
                $url = $this->build_url_from_parts($parsed);
            }
        }

        return $url;
    }

    /**
     * Build URL from parse_url parts
     */
    private function build_url_from_parts($parts) {
        $scheme = isset($parts['scheme']) ? $parts['scheme'] . '://' : '';
        $user = isset($parts['user']) ? $parts['user'] : '';
        $pass = isset($parts['pass']) ? ':' . $parts['pass'] : '';
        $auth = $user !== '' ? $user . $pass . '@' : '';
        $host = isset($parts['host']) ? $parts['host'] : '';
        $port = isset($parts['port']) ? ':' . $parts['port'] : '';
        $path = isset($parts['path']) ? $parts['path'] : '';
        $query = isset($parts['query']) ? '?' . $parts['query'] : '';
        $fragment = isset($parts['fragment']) ? '#' . $parts['fragment'] : '';

        return $scheme . $auth . $host . $port . $path . $query . $fragment;
    }

    /**
     * Map MIME type to file extension
     */
    private function guess_extension_from_mime($mime_type) {
        if (empty($mime_type)) {
            return '';
        }

        $mime_type = strtolower($mime_type);
        $map = array(
            'image/jpeg' => 'jpg',
            'image/jpg' => 'jpg',
            'image/png' => 'png',
            'image/gif' => 'gif',
            'image/webp' => 'webp',
            'image/bmp' => 'bmp',
            'image/tiff' => 'tiff',
            'image/svg+xml' => 'svg',
            'image/avif' => 'avif',
            'image/heic' => 'heic',
            'image/heif' => 'heif'
        );

        return isset($map[$mime_type]) ? $map[$mime_type] : '';
    }

    /**
     * Map file extension to MIME type
     */
    private function guess_mime_from_extension($extension) {
        if (empty($extension)) {
            return 'image/jpeg';
        }

        $extension = strtolower($extension);
        $map = array(
            'jpg' => 'image/jpeg',
            'jpeg' => 'image/jpeg',
            'png' => 'image/png',
            'gif' => 'image/gif',
            'webp' => 'image/webp',
            'bmp' => 'image/bmp',
            'tiff' => 'image/tiff',
            'svg' => 'image/svg+xml',
            'avif' => 'image/avif',
            'heic' => 'image/heic',
            'heif' => 'image/heif'
        );

        return isset($map[$extension]) ? $map[$extension] : 'image/jpeg';
    }
    
    /**
     * Update SEO meta tags for Yoast SEO and Rank Math
     */
    private function update_seo_meta($post_id, $data) {
        // Update Yoast SEO meta
        if (class_exists('WPSEO_Meta')) {
            if (!empty($data['metaTitle'])) {
                update_post_meta($post_id, '_yoast_wpseo_title', sanitize_text_field($data['metaTitle']));
            }
            if (!empty($data['metaDescription'])) {
                update_post_meta($post_id, '_yoast_wpseo_metadesc', sanitize_textarea_field($data['metaDescription']));
            }
            if (!empty($data['focusKeyword'])) {
                update_post_meta($post_id, '_yoast_wpseo_focuskw', sanitize_text_field($data['focusKeyword']));
            }
        }
        
        // Update Rank Math meta
        if (class_exists('RankMath')) {
            if (!empty($data['metaTitle'])) {
                update_post_meta($post_id, 'rank_math_title', sanitize_text_field($data['metaTitle']));
            }
            if (!empty($data['metaDescription'])) {
                update_post_meta($post_id, 'rank_math_description', sanitize_textarea_field($data['metaDescription']));
            }
            if (!empty($data['focusKeyword'])) {
                update_post_meta($post_id, 'rank_math_focus_keyword', sanitize_text_field($data['focusKeyword']));
            }
        }
    }
    
    /**
     * Get or create category based on topic using LLM selection
     */
    private function get_or_create_category($topic) {
        try {
            error_log('Mosallas Publisher: Starting LLM-based category selection for topic: ' . $topic);

            // Get all available categories from WordPress
            $categories = get_categories(array(
                'hide_empty' => false,
                'orderby' => 'name',
                'order' => 'ASC'
            ));

            if (empty($categories)) {
                error_log('Mosallas Publisher: No categories found, will generate new category');
                return $this->generate_short_category($topic, array(), array(
                    'title' => $this->current_article_title ?? '',
                    'content' => $this->current_article_content ?? '',
                    'topic' => $topic
                ));
            }

            // Prepare categories data for API call
            $categories_data = array();
            foreach ($categories as $category) {
                $categories_data[] = array(
                    'id' => $category->term_id,
                    'name' => $category->name,
                    'description' => $category->description
                );
            }

            error_log('Mosallas Publisher: Found ' . count($categories) . ' existing categories');

            // Get article data from the current context
            $article_data = array(
                'title' => $this->current_article_title ?? '',
                'content' => $this->current_article_content ?? '',
                'topic' => $topic
            );

            // First try to select from existing categories
            $selected_category = $this->select_category_from_existing($categories_data, $article_data);

            if ($selected_category && isset($selected_category['categoryId']) && $selected_category['categoryId'] > 0) {
                error_log('Mosallas Publisher: Selected existing category ID: ' . $selected_category['categoryId'] . ' (' . $selected_category['categoryName'] . ')');
                Mosallas_Logger::info('🤖 دسته‌بندی موجود انتخاب شد', array(
                    'category_id' => $selected_category['categoryId'],
                    'category_name' => $selected_category['categoryName'],
                    'reasoning' => $selected_category['reasoning']
                ));
                return array($selected_category['categoryId']);
            }

            error_log('Mosallas Publisher: No suitable existing category found, generating new short category');

            // If no suitable category found, generate a new short category
            return $this->generate_short_category($topic, $categories_data, $article_data);

        } catch (Exception $e) {
            error_log('Mosallas Publisher: LLM category selection failed - ' . $e->getMessage());
            Mosallas_Logger::warning('⚠️ خطا در انتخاب دسته‌بندی توسط هوش مصنوعی', array(
                'error' => $e->getMessage()
            ));
            // Fallback to original logic
            return $this->get_or_create_category_fallback($topic);
        }
    }
    
    /**
     * Original category selection logic as fallback
     */
    private function get_or_create_category_fallback($topic) {
        if (empty($topic)) {
            return array(get_option('default_category'));
        }
        
        // Try to find existing category
        $categories = get_categories(array(
            'hide_empty' => false,
            'search' => $topic
        ));
        
        foreach ($categories as $category) {
            if (stripos($topic, $category->name) !== false || 
                stripos($category->name, $topic) !== false) {
                return array($category->term_id);
            }
        }
        
        // Ensure taxonomy helpers are available when running via REST
        if (!function_exists('wp_insert_term')) {
            require_once ABSPATH . 'wp-admin/includes/taxonomy.php';
        }

        $insert_result = wp_insert_term($topic, 'category');

        if (is_wp_error($insert_result)) {
            error_log('Mosallas Publisher: Failed to insert category - ' . $insert_result->get_error_message());
            return array(get_option('default_category'));
        }

        return array($insert_result['term_id']);
    }
    
    /**
     * Select category using LLM via server API (select from existing)
     */
    private function select_category_from_existing($categories, $article_data) {
        $api_key = get_option('mosallas_api_key');
        $api_base = 'https://mosallas.ir/api/wordpress';
        
        if (empty($api_key)) {
            throw new Exception('API key not configured');
        }
        
        $request_data = array(
            'action' => 'select_category',
            'data' => array(
                'categories' => $categories,
                'articleTitle' => $article_data['title'],
                'articleContent' => $article_data['content'],
                'articleTopic' => $article_data['topic']
            )
        );
        
        error_log('Mosallas Publisher: Calling select_category API with ' . count($categories) . ' categories');
        
        $response = wp_remote_post($api_base . '/webhook', array(
            'headers' => array(
                'X-API-Key' => $api_key,
                'Content-Type' => 'application/json'
            ),
            'body' => json_encode($request_data),
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) {
            throw new Exception('API request failed: ' . $response->get_error_message());
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (!$body || !isset($body['success']) || !$body['success']) {
            throw new Exception('API returned error: ' . json_encode($body));
        }
        
        $category = $body['category'] ?? null;
        error_log('Mosallas Publisher: select_category API returned: ' . json_encode($category));
        
        return $category;
    }
    
    /**
     * Generate short category name using LLM via server API
     */
    private function generate_short_category($topic, $existing_categories, $article_data) {
        $api_key = get_option('mosallas_api_key');
        $api_base = 'https://mosallas.ir/api/wordpress';
        
        if (empty($api_key)) {
            throw new Exception('API key not configured');
        }
        
        $request_data = array(
            'action' => 'generate_category',
            'data' => array(
                'articleTitle' => $article_data['title'],
                'articleContent' => $article_data['content'],
                'articleTopic' => $article_data['topic']
            )
        );
        
        error_log('Mosallas Publisher: Calling generate_category API for topic: ' . $topic);
        
        $response = wp_remote_post($api_base . '/webhook', array(
            'headers' => array(
                'X-API-Key' => $api_key,
                'Content-Type' => 'application/json'
            ),
            'body' => json_encode($request_data),
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) {
            throw new Exception('API request failed: ' . $response->get_error_message());
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (!$body || !isset($body['success']) || !$body['success']) {
            throw new Exception('API returned error: ' . json_encode($body));
        }
        
        $category = $body['category'] ?? null;
        error_log('Mosallas Publisher: generate_category API returned: ' . json_encode($category));
        
        if ($category && isset($category['categoryName'])) {
            $category_name = trim($category['categoryName']);
            
            // Try to find existing category with this name (in case it was created before)
            $existing_category = get_term_by('name', $category_name, 'category');
            
            if ($existing_category) {
                $category_id = $existing_category->term_id;
                error_log('Mosallas Publisher: Found existing generated category: ' . $category_name . ' (ID: ' . $category_id . ')');
            } else {
                // Create new category with the generated short name
                if (!function_exists('wp_insert_term')) {
                    require_once ABSPATH . 'wp-admin/includes/taxonomy.php';
                }
                
                $insert_result = wp_insert_term($category_name, 'category');
                
                if (is_wp_error($insert_result)) {
                    error_log('Mosallas Publisher: Failed to create category "' . $category_name . '": ' . $insert_result->get_error_message());
                    return array(get_option('default_category'));
                }
                
                $category_id = $insert_result['term_id'];
                error_log('Mosallas Publisher: Created new short category: ' . $category_name . ' (ID: ' . $category_id . ')');
            }
            
            Mosallas_Logger::info('🤖 دسته‌بندی کوتاه جدید تولید شد', array(
                'category_id' => $category_id,
                'category_name' => $category_name,
                'reasoning' => $category['reasoning'],
                'topic' => $topic
            ));
            return array($category_id);
        }
        
        // Fallback if generation failed
        error_log('Mosallas Publisher: Category generation failed, using fallback');
        return $this->get_or_create_category_fallback($topic);
    }
    
    /**
     * Add internal links to related posts
     */
    private function add_internal_links($post_id, $existing_posts) {
        if (!is_array($existing_posts) || empty($existing_posts)) {
            return;
        }
        
        $post = get_post($post_id);
        if (!$post) {
            return;
        }
        
        $content = $post->post_content;
        $links_added = 0;
        $max_links = 3;
        
        foreach ($existing_posts as $existing_post) {
            if ($links_added >= $max_links) {
                break;
            }
            
            $keyword = $existing_post['focusKeyword'] ?? '';
            $post_id_link = $existing_post['wpPostId'] ?? 0;
            
            if (empty($keyword) || empty($post_id_link)) {
                continue;
            }
            
            // Check if keyword exists in content (case-insensitive)
            if (stripos($content, $keyword) !== false) {
                $permalink = get_permalink($post_id_link);
                
                // Create the link
                $link = '<a href="' . esc_url($permalink) . '" title="' . esc_attr($keyword) . '">' . 
                        esc_html($keyword) . '</a>';
                
                // Replace first occurrence of the keyword (case-insensitive)
                $content = preg_replace(
                    '/\b(' . preg_quote($keyword, '/') . ')\b/ui',
                    $link,
                    $content,
                    1,
                    $count
                );
                
                if ($count > 0) {
                    $links_added++;
                }
            }
        }
        
        // Update post if links were added
        if ($links_added > 0) {
            wp_update_post(array(
                'ID' => $post_id,
                'post_content' => $content
            ));
        }
    }
    
    /**
     * Get author ID for posts (defaults to first admin user)
     */
    private function get_author_id() {
        $current_user = get_current_user_id();
        
        if ($current_user > 0) {
            return $current_user;
        }
        
        // Get first admin user
        $admin_users = get_users(array(
            'role' => 'administrator',
            'number' => 1
        ));
        
        if (!empty($admin_users)) {
            return $admin_users[0]->ID;
        }
        
        return 1; // Fallback
    }
}



