<?php
/**
 * Mosallas Logger
 * Handles logging of plugin activities
 */

if (!defined('ABSPATH')) {
    exit;
}

class Mosallas_Logger {
    
    const LOG_OPTION = 'mosallas_logs';
    const MAX_LOGS = 500; // Keep last 500 logs
    
    /**
     * Log levels
     */
    const LEVEL_INFO = 'info';
    const LEVEL_SUCCESS = 'success';
    const LEVEL_WARNING = 'warning';
    const LEVEL_ERROR = 'error';
    
    /**
     * Add a log entry
     */
    public static function log($message, $level = self::LEVEL_INFO, $context = array()) {
        $logs = get_option(self::LOG_OPTION, array());
        
        // Create log entry
        $entry = array(
            'id' => uniqid('log_'),
            'timestamp' => current_time('mysql'),
            'level' => $level,
            'message' => $message,
            'context' => $context,
            'user_id' => get_current_user_id(),
            'ip' => self::get_client_ip()
        );
        
        // Add to beginning of array
        array_unshift($logs, $entry);
        
        // Keep only MAX_LOGS entries
        if (count($logs) > self::MAX_LOGS) {
            $logs = array_slice($logs, 0, self::MAX_LOGS);
        }
        
        // Save logs
        update_option(self::LOG_OPTION, $logs);
        
        // Also log to PHP error log if it's an error
        if ($level === self::LEVEL_ERROR) {
            error_log('Mosallas Error: ' . $message . ' | Context: ' . json_encode($context));
        }
    }
    
    /**
     * Log info message
     */
    public static function info($message, $context = array()) {
        self::log($message, self::LEVEL_INFO, $context);
    }
    
    /**
     * Log success message
     */
    public static function success($message, $context = array()) {
        self::log($message, self::LEVEL_SUCCESS, $context);
    }
    
    /**
     * Log warning message
     */
    public static function warning($message, $context = array()) {
        self::log($message, self::LEVEL_WARNING, $context);
    }
    
    /**
     * Log error message
     */
    public static function error($message, $context = array()) {
        self::log($message, self::LEVEL_ERROR, $context);
    }
    
    /**
     * Get all logs
     */
    public static function get_logs($level = null, $search = null, $limit = 100) {
        $logs = get_option(self::LOG_OPTION, array());
        
        // Filter by level
        if ($level && $level !== 'all') {
            $logs = array_filter($logs, function($log) use ($level) {
                return $log['level'] === $level;
            });
        }
        
        // Filter by search term
        if ($search) {
            $logs = array_filter($logs, function($log) use ($search) {
                return stripos($log['message'], $search) !== false ||
                       stripos(json_encode($log['context']), $search) !== false;
            });
        }
        
        // Limit results
        if ($limit > 0) {
            $logs = array_slice($logs, 0, $limit);
        }
        
        return array_values($logs); // Re-index array
    }
    
    /**
     * Clear all logs
     */
    public static function clear_logs() {
        delete_option(self::LOG_OPTION);
        self::info('تمام لاگ‌ها پاک شدند');
    }
    
    /**
     * Get logs count by level
     */
    public static function get_logs_count($level = null) {
        $logs = get_option(self::LOG_OPTION, array());
        
        if ($level && $level !== 'all') {
            $logs = array_filter($logs, function($log) use ($level) {
                return $log['level'] === $level;
            });
        }
        
        return count($logs);
    }
    
    /**
     * Export logs as JSON
     */
    public static function export_logs() {
        $logs = get_option(self::LOG_OPTION, array());
        return json_encode($logs, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    }
    
    /**
     * Get client IP address
     */
    private static function get_client_ip() {
        $ip = '';
        
        if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
            $ip = $_SERVER['HTTP_CLIENT_IP'];
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            $ip = $_SERVER['HTTP_X_FORWARDED_FOR'];
        } else {
            $ip = $_SERVER['REMOTE_ADDR'] ?? '';
        }
        
        return $ip;
    }
    
    /**
     * Format timestamp for display
     */
    public static function format_time($mysql_time) {
        $timestamp = strtotime($mysql_time);
        $now = current_time('timestamp');
        $diff = $now - $timestamp;
        
        if ($diff < 60) {
            return 'لحظاتی پیش';
        } elseif ($diff < 3600) {
            $minutes = floor($diff / 60);
            return $minutes . ' دقیقه پیش';
        } elseif ($diff < 86400) {
            $hours = floor($diff / 3600);
            return $hours . ' ساعت پیش';
        } else {
            return date_i18n('Y/m/d H:i:s', $timestamp);
        }
    }
}
