<?php
/**
 * Mosallas Database Handler
 * Handles database operations for articles and products synchronization
 */

if (!defined('ABSPATH')) {
    exit;
}

class Mosallas_Database {
    
    /**
     * PostgreSQL connection details
     */
    private $db_config;
    private $pdo;
    
    public function __construct() {
        // Get database configuration from WordPress options
        $this->db_config = array(
            'host' => get_option('mosallas_db_host', 'localhost'),
            'port' => get_option('mosallas_db_port', '5432'),
            'dbname' => get_option('mosallas_db_name', 'mosallas'),
            'user' => get_option('mosallas_db_user', ''),
            'password' => get_option('mosallas_db_password', '')
        );
        
        $this->init_connection();
    }
    
    /**
     * Initialize PostgreSQL connection
     */
    private function init_connection() {
        try {
            $dsn = sprintf(
                'pgsql:host=%s;port=%s;dbname=%s',
                $this->db_config['host'],
                $this->db_config['port'],
                $this->db_config['dbname']
            );
            
            $this->pdo = new PDO($dsn, $this->db_config['user'], $this->db_config['password'], [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
            ]);
            
            Mosallas_Logger::log('Database connection established successfully');
        } catch (PDOException $e) {
            Mosallas_Logger::log('Database connection failed: ' . $e->getMessage(), 'error');
            $this->pdo = null;
        }
    }
    
    /**
     * Sync all WordPress articles to database
     */
    public function sync_articles() {
        if (!$this->pdo) {
            Mosallas_Logger::log('Database connection not available for article sync', 'error');
            return false;
        }
        
        $site_id = get_option('mosallas_site_id', wp_generate_uuid4());
        
        // Get all published posts
        $posts = get_posts(array(
            'post_type' => array('post', 'page'),
            'post_status' => 'publish',
            'numberposts' => -1,
            'meta_query' => array(
                array(
                    'key' => '_mosallas_synced',
                    'compare' => 'NOT EXISTS'
                )
            )
        ));
        
        $synced_count = 0;
        
        foreach ($posts as $post) {
            if ($this->save_article_to_db($post, $site_id)) {
                // Mark as synced
                update_post_meta($post->ID, '_mosallas_synced', current_time('mysql'));
                $synced_count++;
            }
        }
        
        Mosallas_Logger::log("Synced {$synced_count} articles to database");
        return $synced_count;
    }
    
    /**
     * Save individual article to database
     */
    public function save_article_to_db($post, $site_id = null) {
        if (!$this->pdo) {
            return false;
        }
        
        if (!$site_id) {
            $site_id = get_option('mosallas_site_id', wp_generate_uuid4());
        }
        
        try {
            // Get SEO meta data
            $meta_title = get_post_meta($post->ID, '_yoast_wpseo_title', true) ?: $post->post_title;
            $meta_description = get_post_meta($post->ID, '_yoast_wpseo_metadesc', true) ?: wp_trim_words($post->post_content, 20);
            $focus_keyword = get_post_meta($post->ID, '_yoast_wpseo_focuskw', true);
            
            // Get featured image
            $featured_image_url = get_the_post_thumbnail_url($post->ID, 'full') ?: null;
            
            // Get categories and tags
            $categories = wp_get_post_categories($post->ID, array('fields' => 'all'));
            $tags = wp_get_post_tags($post->ID, array('fields' => 'all'));
            
            $categories_json = json_encode(array_map(function($cat) {
                return array('id' => $cat->term_id, 'name' => $cat->name, 'slug' => $cat->slug);
            }, $categories));
            
            $tags_json = json_encode(array_map(function($tag) {
                return array('id' => $tag->term_id, 'name' => $tag->name, 'slug' => $tag->slug);
            }, $tags));
            
            // Get author info
            $author = get_userdata($post->post_author);
            
            $sql = "INSERT INTO wordpress_article (
                id, \"siteId\", \"wpPostId\", title, content, excerpt, permalink, status, 
                \"postType\", \"metaTitle\", \"metaDescription\", \"focusKeyword\", 
                \"featuredImageUrl\", categories, tags, \"authorName\", \"authorId\", 
                \"publishedAt\", \"modifiedAt\", \"lastSyncedAt\", \"isActive\", 
                \"createdAt\", \"updatedAt\"
            ) VALUES (
                :id, :siteId, :wpPostId, :title, :content, :excerpt, :permalink, :status,
                :postType, :metaTitle, :metaDescription, :focusKeyword,
                :featuredImageUrl, :categories, :tags, :authorName, :authorId,
                :publishedAt, :modifiedAt, :lastSyncedAt, :isActive,
                :createdAt, :updatedAt
            ) ON CONFLICT (\"siteId\", \"wpPostId\") DO UPDATE SET
                title = EXCLUDED.title,
                content = EXCLUDED.content,
                excerpt = EXCLUDED.excerpt,
                permalink = EXCLUDED.permalink,
                status = EXCLUDED.status,
                \"metaTitle\" = EXCLUDED.\"metaTitle\",
                \"metaDescription\" = EXCLUDED.\"metaDescription\",
                \"focusKeyword\" = EXCLUDED.\"focusKeyword\",
                \"featuredImageUrl\" = EXCLUDED.\"featuredImageUrl\",
                categories = EXCLUDED.categories,
                tags = EXCLUDED.tags,
                \"modifiedAt\" = EXCLUDED.\"modifiedAt\",
                \"lastSyncedAt\" = EXCLUDED.\"lastSyncedAt\",
                \"updatedAt\" = EXCLUDED.\"updatedAt\"";
            
            $stmt = $this->pdo->prepare($sql);
            
            $article_id = $site_id . '_post_' . $post->ID;
            $now = date('Y-m-d H:i:s');
            
            $result = $stmt->execute([
                'id' => $article_id,
                'siteId' => $site_id,
                'wpPostId' => $post->ID,
                'title' => $post->post_title,
                'content' => $post->post_content,
                'excerpt' => $post->post_excerpt ?: wp_trim_words($post->post_content, 55),
                'permalink' => get_permalink($post->ID),
                'status' => $post->post_status,
                'postType' => $post->post_type,
                'metaTitle' => $meta_title,
                'metaDescription' => $meta_description,
                'focusKeyword' => $focus_keyword,
                'featuredImageUrl' => $featured_image_url,
                'categories' => $categories_json,
                'tags' => $tags_json,
                'authorName' => $author ? $author->display_name : '',
                'authorId' => $post->post_author,
                'publishedAt' => $post->post_date,
                'modifiedAt' => $post->post_modified,
                'lastSyncedAt' => $now,
                'isActive' => true,
                'createdAt' => $now,
                'updatedAt' => $now
            ]);
            
            return $result;
            
        } catch (PDOException $e) {
            Mosallas_Logger::log('Failed to save article to database: ' . $e->getMessage(), 'error');
            return false;
        }
    }
    
    /**
     * Sync WooCommerce products to database
     */
    public function sync_products() {
        if (!$this->pdo || !class_exists('WooCommerce')) {
            Mosallas_Logger::log('WooCommerce not available or database connection failed', 'error');
            return false;
        }
        
        $site_id = get_option('mosallas_site_id', wp_generate_uuid4());
        
        // Get all published products
        $products = wc_get_products(array(
            'status' => 'publish',
            'limit' => -1,
            'meta_query' => array(
                array(
                    'key' => '_mosallas_synced',
                    'compare' => 'NOT EXISTS'
                )
            )
        ));
        
        $synced_count = 0;
        
        foreach ($products as $product) {
            if ($this->save_product_to_db($product, $site_id)) {
                // Mark as synced
                update_post_meta($product->get_id(), '_mosallas_synced', current_time('mysql'));
                $synced_count++;
            }
        }
        
        Mosallas_Logger::log("Synced {$synced_count} products to database");
        return $synced_count;
    }
    
    /**
     * Save individual product to database
     */
    public function save_product_to_db($product, $site_id = null) {
        if (!$this->pdo) {
            return false;
        }
        
        if (!$site_id) {
            $site_id = get_option('mosallas_site_id', wp_generate_uuid4());
        }
        
        try {
            // Get product images
            $images = array();
            $image_ids = $product->get_gallery_image_ids();
            if ($product->get_image_id()) {
                array_unshift($image_ids, $product->get_image_id());
            }
            
            foreach ($image_ids as $image_id) {
                $images[] = array(
                    'id' => $image_id,
                    'url' => wp_get_attachment_url($image_id),
                    'alt' => get_post_meta($image_id, '_wp_attachment_image_alt', true)
                );
            }
            
            // Get categories and tags
            $categories = wp_get_post_terms($product->get_id(), 'product_cat');
            $tags = wp_get_post_terms($product->get_id(), 'product_tag');
            
            $categories_json = json_encode(array_map(function($cat) {
                return array('id' => $cat->term_id, 'name' => $cat->name, 'slug' => $cat->slug);
            }, $categories));
            
            $tags_json = json_encode(array_map(function($tag) {
                return array('id' => $tag->term_id, 'name' => $tag->name, 'slug' => $tag->slug);
            }, $tags));
            
            // Get SEO meta
            $meta_title = get_post_meta($product->get_id(), '_yoast_wpseo_title', true) ?: $product->get_name();
            $meta_description = get_post_meta($product->get_id(), '_yoast_wpseo_metadesc', true) ?: wp_trim_words($product->get_description(), 20);
            
            $sql = "INSERT INTO wordpress_product (
                id, \"siteId\", \"wpProductId\", name, description, \"shortDescription\", 
                permalink, status, \"productType\", \"regularPrice\", \"salePrice\", 
                currency, sku, \"stockStatus\", \"stockQuantity\", images, categories, 
                tags, \"metaTitle\", \"metaDescription\", \"publishedAt\", \"modifiedAt\", 
                \"lastSyncedAt\", \"isActive\", \"createdAt\", \"updatedAt\"
            ) VALUES (
                :id, :siteId, :wpProductId, :name, :description, :shortDescription,
                :permalink, :status, :productType, :regularPrice, :salePrice,
                :currency, :sku, :stockStatus, :stockQuantity, :images, :categories,
                :tags, :metaTitle, :metaDescription, :publishedAt, :modifiedAt,
                :lastSyncedAt, :isActive, :createdAt, :updatedAt
            ) ON CONFLICT (\"siteId\", \"wpProductId\") DO UPDATE SET
                name = EXCLUDED.name,
                description = EXCLUDED.description,
                \"shortDescription\" = EXCLUDED.\"shortDescription\",
                permalink = EXCLUDED.permalink,
                status = EXCLUDED.status,
                \"regularPrice\" = EXCLUDED.\"regularPrice\",
                \"salePrice\" = EXCLUDED.\"salePrice\",
                \"stockStatus\" = EXCLUDED.\"stockStatus\",
                \"stockQuantity\" = EXCLUDED.\"stockQuantity\",
                images = EXCLUDED.images,
                categories = EXCLUDED.categories,
                tags = EXCLUDED.tags,
                \"metaTitle\" = EXCLUDED.\"metaTitle\",
                \"metaDescription\" = EXCLUDED.\"metaDescription\",
                \"modifiedAt\" = EXCLUDED.\"modifiedAt\",
                \"lastSyncedAt\" = EXCLUDED.\"lastSyncedAt\",
                \"updatedAt\" = EXCLUDED.\"updatedAt\"";
            
            $stmt = $this->pdo->prepare($sql);
            
            $product_id = $site_id . '_product_' . $product->get_id();
            $now = date('Y-m-d H:i:s');
            
            $result = $stmt->execute([
                'id' => $product_id,
                'siteId' => $site_id,
                'wpProductId' => $product->get_id(),
                'name' => $product->get_name(),
                'description' => $product->get_description(),
                'shortDescription' => $product->get_short_description(),
                'permalink' => $product->get_permalink(),
                'status' => $product->get_status(),
                'productType' => $product->get_type(),
                'regularPrice' => $product->get_regular_price(),
                'salePrice' => $product->get_sale_price(),
                'currency' => get_woocommerce_currency(),
                'sku' => $product->get_sku(),
                'stockStatus' => $product->get_stock_status(),
                'stockQuantity' => $product->get_stock_quantity(),
                'images' => json_encode($images),
                'categories' => $categories_json,
                'tags' => $tags_json,
                'metaTitle' => $meta_title,
                'metaDescription' => $meta_description,
                'publishedAt' => get_post_field('post_date', $product->get_id()),
                'modifiedAt' => get_post_field('post_modified', $product->get_id()),
                'lastSyncedAt' => $now,
                'isActive' => true,
                'createdAt' => $now,
                'updatedAt' => $now
            ]);
            
            return $result;
            
        } catch (PDOException $e) {
            Mosallas_Logger::log('Failed to save product to database: ' . $e->getMessage(), 'error');
            return false;
        }
    }
    
    /**
     * Get articles for internal linking
     */
    public function get_articles_for_linking($query = '', $limit = 10) {
        if (!$this->pdo) {
            return array();
        }
        
        try {
            $site_id = get_option('mosallas_site_id', wp_generate_uuid4());
            
            $sql = "SELECT title, permalink, \"metaDescription\", \"focusKeyword\" 
                    FROM wordpress_article 
                    WHERE \"siteId\" = :siteId AND \"isActive\" = true AND status = 'publish'";
            
            $params = array('siteId' => $site_id);
            
            if (!empty($query)) {
                $sql .= " AND (title ILIKE :query OR content ILIKE :query OR \"focusKeyword\" ILIKE :query)";
                $params['query'] = '%' . $query . '%';
            }
            
            $sql .= " ORDER BY \"publishedAt\" DESC LIMIT :limit";
            $params['limit'] = $limit;
            
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            
            return $stmt->fetchAll();
            
        } catch (PDOException $e) {
            Mosallas_Logger::log('Failed to get articles for linking: ' . $e->getMessage(), 'error');
            return array();
        }
    }
    
    /**
     * Get products for internal linking
     */
    public function get_products_for_linking($query = '', $limit = 10) {
        if (!$this->pdo) {
            return array();
        }
        
        try {
            $site_id = get_option('mosallas_site_id', wp_generate_uuid4());
            
            $sql = "SELECT name, permalink, \"shortDescription\", \"regularPrice\", \"salePrice\" 
                    FROM wordpress_product 
                    WHERE \"siteId\" = :siteId AND \"isActive\" = true AND status = 'publish'";
            
            $params = array('siteId' => $site_id);
            
            if (!empty($query)) {
                $sql .= " AND (name ILIKE :query OR description ILIKE :query OR \"shortDescription\" ILIKE :query)";
                $params['query'] = '%' . $query . '%';
            }
            
            $sql .= " ORDER BY \"publishedAt\" DESC LIMIT :limit";
            $params['limit'] = $limit;
            
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            
            return $stmt->fetchAll();
            
        } catch (PDOException $e) {
            Mosallas_Logger::log('Failed to get products for linking: ' . $e->getMessage(), 'error');
            return array();
        }
    }
    
    /**
     * Schedule automatic sync
     */
    public function schedule_auto_sync() {
        if (!wp_next_scheduled('mosallas_auto_sync')) {
            wp_schedule_event(time(), 'hourly', 'mosallas_auto_sync');
        }
    }
    
    /**
     * Unschedule automatic sync
     */
    public function unschedule_auto_sync() {
        wp_clear_scheduled_hook('mosallas_auto_sync');
    }
}