<?php
/**
 * Mosallas API Handler
 * Handles REST API endpoints for communication with Mosallas.ir
 */

if (!defined('ABSPATH')) {
    exit;
}

class Mosallas_API {
    
    /**
     * Mosallas.ir API base URL
     */
    private $api_base = 'https://mosallas.ir/api/wordpress';
    
    /**
     * Register REST API routes
     */
    public function register_routes() {
        // Endpoint for Mosallas.ir to publish articles
        register_rest_route('mosallas/v1', '/publish', array(
            'methods' => 'POST',
            'callback' => array($this, 'publish_article'),
            'permission_callback' => array($this, 'verify_api_key')
        ));
        
        // Endpoint to get existing posts for internal linking
        register_rest_route('mosallas/v1', '/posts', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_existing_posts'),
            'permission_callback' => array($this, 'verify_api_key')
        ));
        
        // Endpoint to get articles for internal linking from database
        register_rest_route('mosallas/v1', '/internal-links/articles', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_internal_link_articles'),
            'permission_callback' => array($this, 'verify_api_key')
        ));
        
        // Endpoint to get products for internal linking from database
        register_rest_route('mosallas/v1', '/internal-links/products', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_internal_link_products'),
            'permission_callback' => array($this, 'verify_api_key')
        ));
        
        // Endpoint to sync articles to database
        register_rest_route('mosallas/v1', '/sync/articles', array(
            'methods' => 'POST',
            'callback' => array($this, 'sync_articles_to_db'),
            'permission_callback' => array($this, 'verify_api_key')
        ));
        
        // Endpoint to sync products to database
        register_rest_route('mosallas/v1', '/sync/products', array(
            'methods' => 'POST',
            'callback' => array($this, 'sync_products_to_db'),
            'permission_callback' => array($this, 'verify_api_key')
        ));
    }
    
    /**
     * Verify API key from request header
     */
    public function verify_api_key($request) {
        $api_key = $request->get_header('X-API-Key');
        $stored_key = get_option('mosallas_api_key');
        
        if (empty($stored_key)) {
            return new WP_Error(
                'no_api_key',
                __('API Key not configured', 'mosallas'),
                array('status' => 401)
            );
        }
        
        if ($api_key !== $stored_key) {
            return new WP_Error(
                'invalid_api_key',
                __('Invalid API Key', 'mosallas'),
                array('status' => 401)
            );
        }
        
        return true;
    }
    
    /**
     * Publish article from Mosallas.ir
     */
    public function publish_article($request) {
        try {
            $data = $request->get_json_params();
            
            // Log raw incoming data for debugging
            error_log('Mosallas API: Received publish request');
            error_log('Mosallas API: Raw data - ' . print_r($data, true));
            
            // Log incoming request
            Mosallas_Logger::info('📥 درخواست انتشار مقاله دریافت شد', array(
                'article_id' => $data['articleId'] ?? 'unknown',
                'title' => $data['title'] ?? 'unknown',
                'has_content' => !empty($data['content']),
                'content_length' => !empty($data['content']) ? strlen($data['content']) : 0
            ));
            
            // Validate required fields
            if (empty($data['title'])) {
                error_log('Mosallas API: Missing title field');
                Mosallas_Logger::error('❌ فیلد عنوان ارسال نشده است');
                return new WP_REST_Response(array(
                    'success' => false,
                    'error' => __('Missing required field: title', 'mosallas')
                ), 400);
            }
            
            if (empty($data['content'])) {
                error_log('Mosallas API: Missing content field');
                Mosallas_Logger::error('❌ فیلد محتوا ارسال نشده است');
                return new WP_REST_Response(array(
                    'success' => false,
                    'error' => __('Missing required field: content', 'mosallas')
                ), 400);
            }
            
            error_log('Mosallas API: Validation passed, creating publisher instance');
            
            $publisher = new Mosallas_Publisher();
            
            error_log('Mosallas API: Calling create_post');
            $post_id = $publisher->create_post($data);
            
            error_log('Mosallas API: Post created successfully with ID: ' . $post_id);
            
            Mosallas_Logger::success('✅ مقاله با موفقیت منتشر شد', array(
                'post_id' => $post_id,
                'title' => $data['title'],
                'article_id' => $data['articleId'] ?? 'unknown'
            ));
            
            // Notify Mosallas.ir of successful publish
            $this->notify_publish_success($data['articleId'] ?? '', $post_id);
            
            // Collect featured image details
            $thumb_id = get_post_thumbnail_id($post_id);
            $thumb_url = '';
            if ($thumb_id) {
                $src = wp_get_attachment_image_src($thumb_id, 'full');
                if ($src && is_array($src)) {
                    $thumb_url = $src[0];
                }
            }
            
            return new WP_REST_Response(array(
                'success' => true,
                'post_id' => $post_id,
                'post_url' => get_permalink($post_id),
                'edit_url' => get_edit_post_link($post_id, 'raw'),
                'featuredImageId' => $thumb_id,
                'featuredImageUrl' => $thumb_url
            ), 200);
            
        } catch (Exception $e) {
            error_log('Mosallas API: Exception caught - ' . $e->getMessage());
            error_log('Mosallas API: Exception trace - ' . $e->getTraceAsString());
            
            Mosallas_Logger::error('❌ خطا در انتشار مقاله: ' . $e->getMessage(), array(
                'article_id' => isset($data) && isset($data['articleId']) ? $data['articleId'] : 'unknown',
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ));
            
            return new WP_REST_Response(array(
                'success' => false,
                'error' => $e->getMessage(),
                'details' => array(
                    'file' => $e->getFile(),
                    'line' => $e->getLine()
                )
            ), 500);
        } catch (Throwable $t) {
            // Catch fatal errors as well
            error_log('Mosallas API: Fatal error - ' . $t->getMessage());
            error_log('Mosallas API: Fatal error trace - ' . $t->getTraceAsString());
            
            Mosallas_Logger::error('❌ خطای جدی در انتشار مقاله: ' . $t->getMessage(), array(
                'error' => $t->getMessage(),
                'trace' => $t->getTraceAsString()
            ));
            
            return new WP_REST_Response(array(
                'success' => false,
                'error' => 'Internal server error: ' . $t->getMessage(),
                'details' => array(
                    'file' => $t->getFile(),
                    'line' => $t->getLine()
                )
            ), 500);
        }
    }
    
    /**
     * Get existing posts for internal linking
     */
    public function get_existing_posts($request) {
        $args = array(
            'post_type' => 'post',
            'post_status' => 'publish',
            'posts_per_page' => 100,
            'orderby' => 'date',
            'order' => 'DESC'
        );
        
        $posts = get_posts($args);
        $result = array();
        
        foreach ($posts as $post) {
            $result[] = array(
                'id' => $post->ID,
                'title' => $post->post_title,
                'url' => get_permalink($post->ID),
                'focusKeyword' => $this->get_focus_keyword($post->ID)
            );
        }
        
        return new WP_REST_Response(array(
            'success' => true,
            'posts' => $result
        ), 200);
    }
    
    /**
     * Get focus keyword from SEO plugins
     */
    private function get_focus_keyword($post_id) {
        // Try Yoast SEO
        $yoast_keyword = get_post_meta($post_id, '_yoast_wpseo_focuskw', true);
        if (!empty($yoast_keyword)) {
            return $yoast_keyword;
        }
        
        // Try Rank Math
        $rankmath_keyword = get_post_meta($post_id, 'rank_math_focus_keyword', true);
        if (!empty($rankmath_keyword)) {
            return $rankmath_keyword;
        }
        
        // Fallback to post title
        return get_the_title($post_id);
    }
    
    /**
     * Confirm connection with Mosallas.ir on activation
     */
    public function confirm_connection() {
        $api_key = get_option('mosallas_api_key');
        
        if (empty($api_key)) {
            Mosallas_Logger::warning('⚠️ کلید API تنظیم نشده است');
            return false;
        }
        
        Mosallas_Logger::info('🔗 در حال تایید اتصال با Mosallas.ir...', array(
            'site_url' => get_site_url(),
            'site_name' => get_bloginfo('name')
        ));
        
        $response = wp_remote_post($this->api_base . '/webhook', array(
            'headers' => array(
                'X-API-Key' => $api_key,
                'Content-Type' => 'application/json'
            ),
            'body' => json_encode(array(
                'action' => 'confirm_connection',
                'site_url' => get_site_url(),
                'site_name' => get_bloginfo('name')
            )),
            'timeout' => 15
        ));
        
        if (is_wp_error($response)) {
            Mosallas_Logger::error('❌ خطا در اتصال به Mosallas.ir', array(
                'error' => $response->get_error_message()
            ));
            update_option('mosallas_connection_status', 'error');
            update_option('mosallas_connection_error', $response->get_error_message());
            return false;
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (isset($body['success']) && $body['success']) {
            Mosallas_Logger::success('✅ اتصال به Mosallas.ir با موفقیت تایید شد');
            update_option('mosallas_connection_status', 'connected');
            update_option('mosallas_connection_error', '');
            return true;
        }
        
        Mosallas_Logger::error('❌ تایید اتصال با Mosallas.ir ناموفق بود', array(
            'response' => $body
        ));
        update_option('mosallas_connection_status', 'error');
        return false;
    }
    
    /**
     * Notify Mosallas.ir of successful article publication
     */
    private function notify_publish_success($article_id, $post_id) {
        $api_key = get_option('mosallas_api_key');
        
        if (empty($api_key) || empty($article_id)) {
            return;
        }
        
        // Resolve featured image details
        $thumb_id = get_post_thumbnail_id($post_id);
        $thumb_url = '';
        if ($thumb_id) {
            $src = wp_get_attachment_image_src($thumb_id, 'full');
            if ($src && is_array($src)) {
                $thumb_url = $src[0];
            }
        }
        
        wp_remote_post($this->api_base . '/webhook', array(
            'headers' => array(
                'X-API-Key' => $api_key,
                'Content-Type' => 'application/json'
            ),
            'body' => json_encode(array(
                'action' => 'confirm_publish',
                'data' => array(
                    'articleId' => $article_id,
                    'wpPostId' => $post_id,
                    'wpPostUrl' => get_permalink($post_id),
                    'wpEditUrl' => get_edit_post_link($post_id, 'raw'),
                    'featuredImageId' => $thumb_id,
                    'featuredImageUrl' => $thumb_url
                )
            )),
            'timeout' => 15,
            'blocking' => false // Don't wait for response
        ));
    }
    
    /**
     * Get articles for internal linking from database
     */
    public function get_internal_link_articles($request) {
        $database = new Mosallas_Database();
        $query = $request->get_param('query') ?: '';
        $limit = intval($request->get_param('limit') ?: 10);
        
        $articles = $database->get_articles_for_linking($query, $limit);
        
        return new WP_REST_Response(array(
            'success' => true,
            'articles' => $articles
        ), 200);
    }
    
    /**
     * Get products for internal linking from database
     */
    public function get_internal_link_products($request) {
        $database = new Mosallas_Database();
        $query = $request->get_param('query') ?: '';
        $limit = intval($request->get_param('limit') ?: 10);
        
        $products = $database->get_products_for_linking($query, $limit);
        
        return new WP_REST_Response(array(
            'success' => true,
            'products' => $products
        ), 200);
    }
    
    /**
     * Sync articles to database
     */
    public function sync_articles_to_db($request) {
        $database = new Mosallas_Database();
        $synced_count = $database->sync_articles();
        
        return new WP_REST_Response(array(
            'success' => true,
            'synced_count' => $synced_count,
            'message' => "Successfully synced {$synced_count} articles to database"
        ), 200);
    }
    
    /**
     * Sync products to database
     */
    public function sync_products_to_db($request) {
        $database = new Mosallas_Database();
        $synced_count = $database->sync_products();
        
        return new WP_REST_Response(array(
            'success' => true,
            'synced_count' => $synced_count,
            'message' => "Successfully synced {$synced_count} products to database"
        ), 200);
    }
}

